﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/gamelift/GameLift_EXPORTS.h>
#include <aws/gamelift/model/S3Location.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GameLift {
namespace Model {

/**
 * <p>Properties describing a Realtime script.</p> <p> <b>Related actions</b> </p>
 * <p> <a
 * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/reference-awssdk.html#reference-awssdk-resources-fleets">All
 * APIs by task</a> </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/Script">AWS API
 * Reference</a></p>
 */
class Script {
 public:
  AWS_GAMELIFT_API Script() = default;
  AWS_GAMELIFT_API Script(Aws::Utils::Json::JsonView jsonValue);
  AWS_GAMELIFT_API Script& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GAMELIFT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A unique identifier for the Realtime script</p>
   */
  inline const Aws::String& GetScriptId() const { return m_scriptId; }
  inline bool ScriptIdHasBeenSet() const { return m_scriptIdHasBeenSet; }
  template <typename ScriptIdT = Aws::String>
  void SetScriptId(ScriptIdT&& value) {
    m_scriptIdHasBeenSet = true;
    m_scriptId = std::forward<ScriptIdT>(value);
  }
  template <typename ScriptIdT = Aws::String>
  Script& WithScriptId(ScriptIdT&& value) {
    SetScriptId(std::forward<ScriptIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (<a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
   * that is assigned to a Amazon GameLift Servers script resource and uniquely
   * identifies it. ARNs are unique across all Regions. In a GameLift script ARN, the
   * resource ID matches the <i>ScriptId</i> value.</p>
   */
  inline const Aws::String& GetScriptArn() const { return m_scriptArn; }
  inline bool ScriptArnHasBeenSet() const { return m_scriptArnHasBeenSet; }
  template <typename ScriptArnT = Aws::String>
  void SetScriptArn(ScriptArnT&& value) {
    m_scriptArnHasBeenSet = true;
    m_scriptArn = std::forward<ScriptArnT>(value);
  }
  template <typename ScriptArnT = Aws::String>
  Script& WithScriptArn(ScriptArnT&& value) {
    SetScriptArn(std::forward<ScriptArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A descriptive label that is associated with a script. Script names do not
   * need to be unique.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  Script& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version information that is associated with a build or script. Version
   * strings do not need to be unique.</p>
   */
  inline const Aws::String& GetVersion() const { return m_version; }
  inline bool VersionHasBeenSet() const { return m_versionHasBeenSet; }
  template <typename VersionT = Aws::String>
  void SetVersion(VersionT&& value) {
    m_versionHasBeenSet = true;
    m_version = std::forward<VersionT>(value);
  }
  template <typename VersionT = Aws::String>
  Script& WithVersion(VersionT&& value) {
    SetVersion(std::forward<VersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The file size of the uploaded Realtime script, expressed in bytes. When files
   * are uploaded from an S3 location, this value remains at "0".</p>
   */
  inline long long GetSizeOnDisk() const { return m_sizeOnDisk; }
  inline bool SizeOnDiskHasBeenSet() const { return m_sizeOnDiskHasBeenSet; }
  inline void SetSizeOnDisk(long long value) {
    m_sizeOnDiskHasBeenSet = true;
    m_sizeOnDisk = value;
  }
  inline Script& WithSizeOnDisk(long long value) {
    SetSizeOnDisk(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A time stamp indicating when this data object was created. Format is a number
   * expressed in Unix time as milliseconds (for example
   * <code>"1469498468.057"</code>).</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  inline bool CreationTimeHasBeenSet() const { return m_creationTimeHasBeenSet; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  Script& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The location of the Amazon S3 bucket where a zipped file containing your
   * Realtime scripts is stored. The storage location must specify the Amazon S3
   * bucket name, the zip file name (the "key"), and a role ARN that allows Amazon
   * GameLift Servers to access the Amazon S3 storage location. The S3 bucket must be
   * in the same Region where you want to create a new script. By default, Amazon
   * GameLift Servers uploads the latest version of the zip file; if you have S3
   * object versioning turned on, you can use the <code>ObjectVersion</code>
   * parameter to specify an earlier version. </p>
   */
  inline const S3Location& GetStorageLocation() const { return m_storageLocation; }
  inline bool StorageLocationHasBeenSet() const { return m_storageLocationHasBeenSet; }
  template <typename StorageLocationT = S3Location>
  void SetStorageLocation(StorageLocationT&& value) {
    m_storageLocationHasBeenSet = true;
    m_storageLocation = std::forward<StorageLocationT>(value);
  }
  template <typename StorageLocationT = S3Location>
  Script& WithStorageLocation(StorageLocationT&& value) {
    SetStorageLocation(std::forward<StorageLocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Node.js version used for execution of your Realtime script. The valid
   * values are <code>10.x | 24.x</code>. By default, <code>NodeJsVersion</code> is
   * <code>10.x</code>. This value cannot be updated later. </p>
   */
  inline const Aws::String& GetNodeJsVersion() const { return m_nodeJsVersion; }
  inline bool NodeJsVersionHasBeenSet() const { return m_nodeJsVersionHasBeenSet; }
  template <typename NodeJsVersionT = Aws::String>
  void SetNodeJsVersion(NodeJsVersionT&& value) {
    m_nodeJsVersionHasBeenSet = true;
    m_nodeJsVersion = std::forward<NodeJsVersionT>(value);
  }
  template <typename NodeJsVersionT = Aws::String>
  Script& WithNodeJsVersion(NodeJsVersionT&& value) {
    SetNodeJsVersion(std::forward<NodeJsVersionT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_scriptId;

  Aws::String m_scriptArn;

  Aws::String m_name;

  Aws::String m_version;

  long long m_sizeOnDisk{0};

  Aws::Utils::DateTime m_creationTime{};

  S3Location m_storageLocation;

  Aws::String m_nodeJsVersion;
  bool m_scriptIdHasBeenSet = false;
  bool m_scriptArnHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_versionHasBeenSet = false;
  bool m_sizeOnDiskHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_storageLocationHasBeenSet = false;
  bool m_nodeJsVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
