/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.qpid.protonj2.types.transport;

import static org.junit.jupiter.api.Assertions.assertArrayEquals;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import org.apache.qpid.protonj2.types.Symbol;
import org.junit.jupiter.api.Test;

public class OpenTest {

    @Test
    public void testGetPerformativeType() {
        assertEquals(Performative.PerformativeType.OPEN, new Open().getPerformativeType());
    }

    @Test
    public void testToStringOnFreshInstance() {
        assertNotNull(new Open().toString());
    }

    @Test
    public void testMaxFrameSizeLimitsImposed() {
        Open open = new Open();

        assertThrows(IllegalArgumentException.class, () -> open.setMaxFrameSize(Long.MAX_VALUE));
        assertThrows(IllegalArgumentException.class, () -> open.setMaxFrameSize(-1l));
    }

    @Test
    public void testSetGetOfOutgoingLocales() {
        Open open = new Open();
        Symbol[] expected = { Symbol.valueOf("1"), Symbol.valueOf("2") };

        open.setOutgoingLocales(Symbol.valueOf("1"), Symbol.valueOf("2"));

        assertArrayEquals(expected, open.getOutgoingLocales());
        assertNotNull(open.toString());
    }

    @Test
    public void testSetGetOfIncomingLocales() {
        Open open = new Open();
        Symbol[] expected = { Symbol.valueOf("1"), Symbol.valueOf("2") };

        open.setIncomingLocales(Symbol.valueOf("1"), Symbol.valueOf("2"));

        assertArrayEquals(expected, open.getIncomingLocales());
        assertNotNull(open.toString());
    }

    @Test
    public void testSetChannelMaxFromShort() {
        Open open = new Open();

        open.setChannelMax((short) 65535);
        assertEquals(65535, open.getChannelMax());
        open.setChannelMax((short) -1);
        assertEquals(65535, open.getChannelMax());
        open.setChannelMax((short) 0);
        assertEquals(0, open.getChannelMax());
    }

    @Test
    public void testInitialState() {
        Open open = new Open();

        assertEquals(1, open.getElementCount());
        assertFalse(open.isEmpty());
        assertFalse(open.hasChannelMax());
        assertTrue(open.hasContainerId());
        assertFalse(open.hasDesiredCapabilities());
        assertFalse(open.hasHostname());
        assertFalse(open.hasIdleTimeout());
        assertFalse(open.hasIncomingLocales());
        assertFalse(open.hasMaxFrameSize());
        assertFalse(open.hasOfferedCapabilities());
        assertFalse(open.hasOutgoingLocales());
        assertFalse(open.hasProperties());
    }

    @Test
    public void testChannelMaxAlwaysPresentOnceSet() {
        Open open = new Open();

        assertFalse(open.hasChannelMax());
        open.setChannelMax(0);
        assertTrue(open.hasChannelMax());
        open.setChannelMax(32767);
        assertTrue(open.hasChannelMax());
        open.setChannelMax(65535);
        assertTrue(open.hasChannelMax());
    }

    @Test
    public void testCopyFromNew() {
        Open original = new Open();
        Open copy = original.copy();

        assertFalse(original.isEmpty());
        assertFalse(copy.isEmpty());

        assertEquals(1, original.getElementCount());
        assertEquals(1, copy.getElementCount());
    }
}
