// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

use std::pin::Pin;
use std::task::Context;
use std::task::Poll;

use bytes::Bytes;
use futures::SinkExt;

use crate::raw::*;
use crate::*;

/// FuturesBytesSink is the adapter of [`futures::Sink`] generated by [`Writer::into_bytes_sink`].
///
/// Users can use this adapter in cases where they need to use [`futures::Sink`] trait. FuturesBytesSink
/// reuses the same concurrent and chunk settings from [`Writer`].
///
/// FuturesBytesSink also implements [`Unpin`], [`Send`] and [`Sync`].
pub struct FuturesBytesSink {
    sink: BufferSink,
}

impl FuturesBytesSink {
    /// Create a new sink from a [`oio::Writer`].
    #[inline]
    pub(crate) fn new(w: WriteGenerator<oio::Writer>) -> Self {
        FuturesBytesSink {
            sink: BufferSink::new(w),
        }
    }
}

impl futures::Sink<Bytes> for FuturesBytesSink {
    type Error = std::io::Error;

    fn poll_ready(
        mut self: Pin<&mut Self>,
        cx: &mut Context<'_>,
    ) -> Poll<Result<(), std::io::Error>> {
        self.sink.poll_ready_unpin(cx).map_err(format_std_io_error)
    }

    fn start_send(mut self: Pin<&mut Self>, item: Bytes) -> Result<(), std::io::Error> {
        self.sink
            .start_send_unpin(Buffer::from(item))
            .map_err(format_std_io_error)
    }

    fn poll_flush(
        mut self: Pin<&mut Self>,
        cx: &mut Context<'_>,
    ) -> Poll<Result<(), std::io::Error>> {
        self.sink.poll_flush_unpin(cx).map_err(format_std_io_error)
    }

    fn poll_close(
        mut self: Pin<&mut Self>,
        cx: &mut Context<'_>,
    ) -> Poll<Result<(), std::io::Error>> {
        self.sink.poll_close_unpin(cx).map_err(format_std_io_error)
    }
}

#[cfg(test)]
mod tests {
    use std::sync::Arc;

    use super::*;
    use crate::raw::MaybeSend;

    #[tokio::test]
    async fn test_trait() {
        let op = Operator::via_iter(Scheme::Memory, []).unwrap();

        let acc = op.into_inner();
        let ctx = Arc::new(WriteContext::new(
            acc,
            "test".to_string(),
            OpWrite::new(),
            OpWriter::new().with_chunk(1),
        ));
        let write_gen = WriteGenerator::create(ctx).await.unwrap();

        let v = FuturesBytesSink::new(write_gen);

        let _: Box<dyn Unpin + MaybeSend + Sync + 'static> = Box::new(v);
    }
}
