/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package migrations

import (
	"github.com/apache/answer/internal/entity"
	"github.com/apache/answer/internal/service/permission"
)

const (
	defaultSEORobotTxt = `User-agent: *
Disallow: /admin
Disallow: /search
Disallow: /install
Disallow: /review
Disallow: /users/login
Disallow: /users/register
Disallow: /users/account-recovery
Disallow: /users/oauth/*
Disallow: /users/*/*
Disallow: /answer/api
Disallow: /*?code*
Disallow: /swagger/*

Sitemap: `
)

var (
	tables = []any{
		&entity.Activity{},
		&entity.Answer{},
		&entity.Collection{},
		&entity.CollectionGroup{},
		&entity.Comment{},
		&entity.Config{},
		&entity.Meta{},
		&entity.Notification{},
		&entity.Question{},
		&entity.QuestionLink{},
		&entity.Report{},
		&entity.Revision{},
		&entity.SiteInfo{},
		&entity.Tag{},
		&entity.TagRel{},
		&entity.Uniqid{},
		&entity.User{},
		&entity.Version{},
		&entity.Role{},
		&entity.RolePowerRel{},
		&entity.Power{},
		&entity.UserRoleRel{},
		&entity.PluginConfig{},
		&entity.UserExternalLogin{},
		&entity.UserNotificationConfig{},
		&entity.PluginUserConfig{},
		&entity.Review{},
		&entity.Badge{},
		&entity.BadgeGroup{},
		&entity.BadgeAward{},
		&entity.FileRecord{},
		&entity.PluginKVStorage{},
		&entity.APIKey{},
		&entity.AIConversation{},
		&entity.AIConversationRecord{},
	}

	roles = []*entity.Role{
		{ID: 1, Name: "User", Description: "Default with no special access."},
		{ID: 2, Name: "Admin", Description: "Have the full power to access the site."},
		{ID: 3, Name: "Moderator", Description: "Has access to all posts except admin settings."},
	}

	powers = []*entity.Power{
		{ID: 1, Name: "admin access", PowerType: permission.AdminAccess, Description: "admin access"},
		{ID: 2, Name: "question add", PowerType: permission.QuestionAdd, Description: "question add"},
		{ID: 3, Name: "question edit", PowerType: permission.QuestionEdit, Description: "question edit"},
		{ID: 4, Name: "question edit without review", PowerType: permission.QuestionEditWithoutReview, Description: "question edit without review"},
		{ID: 5, Name: "question delete", PowerType: permission.QuestionDelete, Description: "question delete"},
		{ID: 6, Name: "question close", PowerType: permission.QuestionClose, Description: "question close"},
		{ID: 7, Name: "question reopen", PowerType: permission.QuestionReopen, Description: "question reopen"},
		{ID: 8, Name: "question vote up", PowerType: permission.QuestionVoteUp, Description: "question vote up"},
		{ID: 9, Name: "question vote down", PowerType: permission.QuestionVoteDown, Description: "question vote down"},
		{ID: 10, Name: "answer add", PowerType: permission.AnswerAdd, Description: "answer add"},
		{ID: 11, Name: "answer edit", PowerType: permission.AnswerEdit, Description: "answer edit"},
		{ID: 12, Name: "answer edit without review", PowerType: permission.AnswerEditWithoutReview, Description: "answer edit without review"},
		{ID: 13, Name: "answer delete", PowerType: permission.AnswerDelete, Description: "answer delete"},
		{ID: 14, Name: "answer accept", PowerType: permission.AnswerAccept, Description: "answer accept"},
		{ID: 15, Name: "answer vote up", PowerType: permission.AnswerVoteUp, Description: "answer vote up"},
		{ID: 16, Name: "answer vote down", PowerType: permission.AnswerVoteDown, Description: "answer vote down"},
		{ID: 17, Name: "comment add", PowerType: permission.CommentAdd, Description: "comment add"},
		{ID: 18, Name: "comment edit", PowerType: permission.CommentEdit, Description: "comment edit"},
		{ID: 19, Name: "comment delete", PowerType: permission.CommentDelete, Description: "comment delete"},
		{ID: 20, Name: "comment vote up", PowerType: permission.CommentVoteUp, Description: "comment vote up"},
		{ID: 21, Name: "comment vote down", PowerType: permission.CommentVoteDown, Description: "comment vote down"},
		{ID: 22, Name: "report add", PowerType: permission.ReportAdd, Description: "report add"},
		{ID: 23, Name: "tag add", PowerType: permission.TagAdd, Description: "tag add"},
		{ID: 24, Name: "tag edit", PowerType: permission.TagEdit, Description: "tag edit"},
		{ID: 25, Name: "tag edit without review", PowerType: permission.TagEditWithoutReview, Description: "tag edit without review"},
		{ID: 26, Name: "tag edit slug name", PowerType: permission.TagEditSlugName, Description: "tag edit slug name"},
		{ID: 27, Name: "tag delete", PowerType: permission.TagDelete, Description: "tag delete"},
		{ID: 28, Name: "tag synonym", PowerType: permission.TagSynonym, Description: "tag synonym"},
		{ID: 29, Name: "link url limit", PowerType: permission.LinkUrlLimit, Description: "link url limit"},
		{ID: 30, Name: "vote detail", PowerType: permission.VoteDetail, Description: "vote detail"},
		{ID: 31, Name: "answer audit", PowerType: permission.AnswerAudit, Description: "answer audit"},
		{ID: 32, Name: "question audit", PowerType: permission.QuestionAudit, Description: "question audit"},
		{ID: 33, Name: "tag audit", PowerType: permission.TagAudit, Description: "tag audit"},
		{ID: 34, Name: "question pin", PowerType: permission.QuestionPin, Description: "top the question"},
		{ID: 35, Name: "question hide", PowerType: permission.QuestionHide, Description: "hide  the question"},
		{ID: 36, Name: "question unpin", PowerType: permission.QuestionUnPin, Description: "untop the question"},
		{ID: 37, Name: "question show", PowerType: permission.QuestionShow, Description: "show the question"},
		{ID: 38, Name: "invite someone to answer", PowerType: permission.AnswerInviteSomeoneToAnswer, Description: "invite someone to answer"},
		{ID: 39, Name: "recover answer", PowerType: permission.AnswerUnDelete, Description: "recover deleted answer"},
		{ID: 40, Name: "recover question", PowerType: permission.QuestionUnDelete, Description: "recover deleted question"},
		{ID: 41, Name: "recover tag", PowerType: permission.TagUnDelete, Description: "recover deleted tag"},
	}

	rolePowerRels = []*entity.RolePowerRel{
		{RoleID: 2, PowerType: permission.AdminAccess},
		{RoleID: 2, PowerType: permission.QuestionAdd},
		{RoleID: 2, PowerType: permission.QuestionEdit},
		{RoleID: 2, PowerType: permission.QuestionEditWithoutReview},
		{RoleID: 2, PowerType: permission.QuestionDelete},
		{RoleID: 2, PowerType: permission.QuestionClose},
		{RoleID: 2, PowerType: permission.QuestionReopen},
		{RoleID: 2, PowerType: permission.QuestionVoteUp},
		{RoleID: 2, PowerType: permission.QuestionVoteDown},
		{RoleID: 2, PowerType: permission.AnswerAdd},
		{RoleID: 2, PowerType: permission.AnswerEdit},
		{RoleID: 2, PowerType: permission.AnswerEditWithoutReview},
		{RoleID: 2, PowerType: permission.AnswerDelete},
		{RoleID: 2, PowerType: permission.AnswerAccept},
		{RoleID: 2, PowerType: permission.AnswerVoteUp},
		{RoleID: 2, PowerType: permission.AnswerVoteDown},
		{RoleID: 2, PowerType: permission.CommentAdd},
		{RoleID: 2, PowerType: permission.CommentEdit},
		{RoleID: 2, PowerType: permission.CommentDelete},
		{RoleID: 2, PowerType: permission.CommentVoteUp},
		{RoleID: 2, PowerType: permission.CommentVoteDown},
		{RoleID: 2, PowerType: permission.ReportAdd},
		{RoleID: 2, PowerType: permission.TagAdd},
		{RoleID: 2, PowerType: permission.TagEdit},
		{RoleID: 2, PowerType: permission.TagEditSlugName},
		{RoleID: 2, PowerType: permission.TagEditWithoutReview},
		{RoleID: 2, PowerType: permission.TagDelete},
		{RoleID: 2, PowerType: permission.TagSynonym},
		{RoleID: 2, PowerType: permission.LinkUrlLimit},
		{RoleID: 2, PowerType: permission.VoteDetail},
		{RoleID: 2, PowerType: permission.AnswerAudit},
		{RoleID: 2, PowerType: permission.QuestionAudit},
		{RoleID: 2, PowerType: permission.TagAudit},
		{RoleID: 2, PowerType: permission.TagUseReservedTag},
		{RoleID: 2, PowerType: permission.QuestionPin},
		{RoleID: 2, PowerType: permission.QuestionHide},
		{RoleID: 2, PowerType: permission.QuestionUnPin},
		{RoleID: 2, PowerType: permission.QuestionShow},
		{RoleID: 2, PowerType: permission.AnswerInviteSomeoneToAnswer},
		{RoleID: 2, PowerType: permission.AnswerUnDelete},
		{RoleID: 2, PowerType: permission.QuestionUnDelete},
		{RoleID: 2, PowerType: permission.TagUnDelete},

		{RoleID: 3, PowerType: permission.QuestionAdd},
		{RoleID: 3, PowerType: permission.QuestionEdit},
		{RoleID: 3, PowerType: permission.QuestionEditWithoutReview},
		{RoleID: 3, PowerType: permission.QuestionDelete},
		{RoleID: 3, PowerType: permission.QuestionClose},
		{RoleID: 3, PowerType: permission.QuestionReopen},
		{RoleID: 3, PowerType: permission.QuestionVoteUp},
		{RoleID: 3, PowerType: permission.QuestionVoteDown},
		{RoleID: 3, PowerType: permission.AnswerAdd},
		{RoleID: 3, PowerType: permission.AnswerEdit},
		{RoleID: 3, PowerType: permission.AnswerEditWithoutReview},
		{RoleID: 3, PowerType: permission.AnswerDelete},
		{RoleID: 3, PowerType: permission.AnswerAccept},
		{RoleID: 3, PowerType: permission.AnswerVoteUp},
		{RoleID: 3, PowerType: permission.AnswerVoteDown},
		{RoleID: 3, PowerType: permission.CommentAdd},
		{RoleID: 3, PowerType: permission.CommentEdit},
		{RoleID: 3, PowerType: permission.CommentDelete},
		{RoleID: 3, PowerType: permission.CommentVoteUp},
		{RoleID: 3, PowerType: permission.CommentVoteDown},
		{RoleID: 3, PowerType: permission.ReportAdd},
		{RoleID: 3, PowerType: permission.TagAdd},
		{RoleID: 3, PowerType: permission.TagEdit},
		{RoleID: 3, PowerType: permission.TagEditSlugName},
		{RoleID: 3, PowerType: permission.TagEditWithoutReview},
		{RoleID: 3, PowerType: permission.TagDelete},
		{RoleID: 3, PowerType: permission.TagSynonym},
		{RoleID: 3, PowerType: permission.LinkUrlLimit},
		{RoleID: 3, PowerType: permission.VoteDetail},
		{RoleID: 3, PowerType: permission.AnswerAudit},
		{RoleID: 3, PowerType: permission.QuestionAudit},
		{RoleID: 3, PowerType: permission.TagAudit},
		{RoleID: 3, PowerType: permission.TagUseReservedTag},
		{RoleID: 3, PowerType: permission.QuestionPin},
		{RoleID: 3, PowerType: permission.QuestionHide},
		{RoleID: 3, PowerType: permission.QuestionUnPin},
		{RoleID: 3, PowerType: permission.QuestionShow},
		{RoleID: 3, PowerType: permission.AnswerInviteSomeoneToAnswer},
		{RoleID: 3, PowerType: permission.AnswerUnDelete},
		{RoleID: 3, PowerType: permission.QuestionUnDelete},
		{RoleID: 3, PowerType: permission.TagUnDelete},
	}

	adminUserRoleRel = &entity.UserRoleRel{
		UserID: "1",
		RoleID: 2,
	}

	defaultConfigTable = []*entity.Config{
		{ID: 1, Key: "answer.accepted", Value: `15`},
		{ID: 2, Key: "answer.voted_up", Value: `10`},
		{ID: 3, Key: "question.voted_up", Value: `10`},
		{ID: 4, Key: "tag.edit_accepted", Value: `2`},
		{ID: 5, Key: "answer.accept", Value: `2`},
		{ID: 6, Key: "answer.voted_down_cancel", Value: `2`},
		{ID: 7, Key: "question.voted_down_cancel", Value: `2`},
		{ID: 8, Key: "answer.vote_down_cancel", Value: `1`},
		{ID: 9, Key: "question.vote_down_cancel", Value: `1`},
		{ID: 10, Key: "user.activated", Value: `1`},
		{ID: 11, Key: "edit.accepted", Value: `2`},
		{ID: 12, Key: "answer.vote_down", Value: `-1`},
		{ID: 13, Key: "question.voted_down", Value: `-2`},
		{ID: 14, Key: "answer.voted_down", Value: `-2`},
		{ID: 15, Key: "answer.accept_cancel", Value: `-2`},
		{ID: 16, Key: "answer.deleted", Value: `-5`},
		{ID: 17, Key: "question.voted_up_cancel", Value: `-10`},
		{ID: 18, Key: "answer.voted_up_cancel", Value: `-10`},
		{ID: 19, Key: "answer.accepted_cancel", Value: `-15`},
		{ID: 20, Key: "object.reported", Value: `-100`},
		{ID: 21, Key: "edit.rejected", Value: `-2`},
		{ID: 22, Key: "daily_rank_limit", Value: `200`},
		{ID: 23, Key: "daily_rank_limit.exclude", Value: `["answer.accepted"]`},
		{ID: 24, Key: "user.follow", Value: `0`},
		{ID: 25, Key: "comment.vote_up", Value: `0`},
		{ID: 26, Key: "comment.vote_up_cancel", Value: `0`},
		{ID: 27, Key: "question.vote_down", Value: `0`},
		{ID: 28, Key: "question.vote_up", Value: `0`},
		{ID: 29, Key: "question.vote_up_cancel", Value: `0`},
		{ID: 30, Key: "answer.vote_up", Value: `0`},
		{ID: 31, Key: "answer.vote_up_cancel", Value: `0`},
		{ID: 32, Key: "question.follow", Value: `0`},
		{ID: 33, Key: "email.config", Value: `{"from_name":"","from_email":"","smtp_host":"","smtp_port":465,"smtp_password":"","smtp_username":"","smtp_authentication":true,"encryption":"","register_title":"[{{.SiteName}}] Confirm your new account","register_body":"Welcome to {{.SiteName}}<br><br>\n\nClick the following link to confirm and activate your new account:<br>\n<a href='{{.RegisterUrl}}' target='_blank'>{{.RegisterUrl}}</a><br><br>\n\nIf the above link is not clickable, try copying and pasting it into the address bar of your web browser.\n","pass_reset_title":"[{{.SiteName }}] Password reset","pass_reset_body":"Somebody asked to reset your password on [{{.SiteName}}].<br><br>\n\nIf it was not you, you can safely ignore this email.<br><br>\n\nClick the following link to choose a new password:<br>\n<a href='{{.PassResetUrl}}' target='_blank'>{{.PassResetUrl}}</a>\n","change_title":"[{{.SiteName}}] Confirm your new email address","change_body":"Confirm your new email address for {{.SiteName}}  by clicking on the following link:<br><br>\n\n<a href='{{.ChangeEmailUrl}}' target='_blank'>{{.ChangeEmailUrl}}</a><br><br>\n\nIf you did not request this change, please ignore this email.\n","test_title":"[{{.SiteName}}] Test Email","test_body":"This is a test email.","new_answer_title":"[{{.SiteName}}] {{.DisplayName}} answered your question","new_answer_body":"<strong><a href='{{.AnswerUrl}}'>{{.QuestionTitle}}</a></strong><br><br>\n\n<small>{{.DisplayName}}:</small><br>\n<blockquote>{{.AnswerSummary}}</blockquote><br>\n<a href='{{.AnswerUrl}}'>View it on {{.SiteName}}</a><br><br>\n\n<small>You are receiving this because you authored the thread. <a href='{{.UnsubscribeUrl}}'>Unsubscribe</a></small>","new_comment_title":"[{{.SiteName}}] {{.DisplayName}} commented on your post","new_comment_body":"<strong><a href='{{.CommentUrl}}'>{{.QuestionTitle}}</a></strong><br><br>\n\n<small>{{.DisplayName}}:</small><br>\n<blockquote>{{.CommentSummary}}</blockquote><br>\n<a href='{{.CommentUrl}}'>View it on {{.SiteName}}</a><br><br>\n\n<small>You are receiving this because you authored the thread. <a href='{{.UnsubscribeUrl}}'>Unsubscribe</a></small>"}`},
		{ID: 35, Key: "tag.follow", Value: `0`},
		{ID: 36, Key: "rank.question.add", Value: `1`},
		{ID: 37, Key: "rank.question.edit", Value: `200`},
		{ID: 38, Key: "rank.question.delete", Value: `-1`},
		{ID: 39, Key: "rank.question.vote_up", Value: `15`},
		{ID: 40, Key: "rank.question.vote_down", Value: `125`},
		{ID: 41, Key: "rank.answer.add", Value: `1`},
		{ID: 42, Key: "rank.answer.edit", Value: `200`},
		{ID: 43, Key: "rank.answer.delete", Value: `-1`},
		{ID: 44, Key: "rank.answer.accept", Value: `-1`},
		{ID: 45, Key: "rank.answer.vote_up", Value: `15`},
		{ID: 46, Key: "rank.answer.vote_down", Value: `125`},
		{ID: 47, Key: "rank.comment.add", Value: `1`},
		{ID: 48, Key: "rank.comment.edit", Value: `-1`},
		{ID: 49, Key: "rank.comment.delete", Value: `-1`},
		{ID: 50, Key: "rank.report.add", Value: `1`},
		{ID: 51, Key: "rank.tag.add", Value: `1500`},
		{ID: 52, Key: "rank.tag.edit", Value: `100`},
		{ID: 53, Key: "rank.tag.delete", Value: `-1`},
		{ID: 54, Key: "rank.tag.synonym", Value: `20000`},
		{ID: 55, Key: "rank.link.url_limit", Value: `10`},
		{ID: 56, Key: "rank.vote.detail", Value: `0`},
		{ID: 57, Key: "reason.spam", Value: `{"name":"spam","description":"This post is an advertisement, or vandalism. It is not useful or relevant to the current topic."}`},
		{ID: 58, Key: "reason.rude_or_abusive", Value: `{"name":"rude or abusive","description":"A reasonable person would find this content inappropriate for respectful discourse."}`},
		{ID: 59, Key: "reason.something", Value: `{"name":"something else","description":"This post requires staff attention for another reason not listed above.","content_type":"textarea"}`},
		{ID: 60, Key: "reason.a_duplicate", Value: `{"name":"a duplicate","description":"This question has been asked before and already has an answer.","content_type":"text"}`},
		{ID: 61, Key: "reason.not_a_answer", Value: `{"name":"not a answer","description":"This was posted as an answer, but it does not attempt to answer the question. It should possibly be an edit, a comment, another question, or deleted altogether.","content_type":""}`},
		{ID: 62, Key: "reason.no_longer_needed", Value: `{"name":"no longer needed","description":"This comment is outdated, conversational or not relevant to this post."}`},
		{ID: 63, Key: "reason.community_specific", Value: `{"name":"a community-specific reason","description":"This question doesn't meet a community guideline."}`},
		{ID: 64, Key: "reason.not_clarity", Value: `{"name":"needs details or clarity","description":"This question currently includes multiple questions in one. It should focus on one problem only."}`},
		{ID: 65, Key: "reason.normal", Value: `{"name":"normal","description":"A normal post available to everyone."}`},
		{ID: 66, Key: "reason.normal.user", Value: `{"name":"normal","description":"A normal user can ask and answer questions."}`},
		{ID: 67, Key: "reason.closed", Value: `{"name":"closed","description":"A closed question can't answer, but still can edit, vote and comment."}`},
		{ID: 68, Key: "reason.deleted", Value: `{"name":"deleted","description":"All reputation gained and lost will be restored."}`},
		{ID: 69, Key: "reason.deleted.user", Value: `{"name":"deleted","description":"Delete profile, authentication associations."}`},
		{ID: 70, Key: "reason.suspended", Value: `{"name":"suspended","description":"A suspended user can't log in."}`},
		{ID: 71, Key: "reason.inactive", Value: `{"name":"inactive","description":"An inactive user must re-validate their email."}`},
		{ID: 72, Key: "reason.looks_ok", Value: `{"name":"looks ok","description":"This post is good as-is and not low quality."}`},
		{ID: 73, Key: "reason.needs_edit", Value: `{"name":"needs edit, and I did it","description":"Improve and correct problems with this post yourself."}`},
		{ID: 74, Key: "reason.needs_close", Value: `{"name":"needs close","description":"A closed question can't answer, but still can edit, vote and comment."}`},
		{ID: 75, Key: "reason.needs_delete", Value: `{"name":"needs delete","description":"All reputation gained and lost will be restored."}`},
		{ID: 76, Key: "question.flag.reasons", Value: `["reason.spam","reason.rude_or_abusive","reason.something","reason.a_duplicate"]`},
		{ID: 77, Key: "answer.flag.reasons", Value: `["reason.spam","reason.rude_or_abusive","reason.something","reason.not_a_answer"]`},
		{ID: 78, Key: "comment.flag.reasons", Value: `["reason.spam","reason.rude_or_abusive","reason.something","reason.no_longer_needed"]`},
		{ID: 79, Key: "question.close.reasons", Value: `["reason.a_duplicate","reason.community_specific","reason.not_clarity","reason.something"]`},
		{ID: 80, Key: "question.status.reasons", Value: `["reason.normal","reason.closed","reason.deleted"]`},
		{ID: 81, Key: "answer.status.reasons", Value: `["reason.normal","reason.deleted"]`},
		{ID: 82, Key: "comment.status.reasons", Value: `["reason.normal","reason.deleted"]`},
		{ID: 83, Key: "user.status.reasons", Value: `["reason.normal.user","reason.suspended","reason.deleted.user","reason.inactive"]`},
		{ID: 84, Key: "question.review.reasons", Value: `["reason.looks_ok","reason.needs_edit","reason.needs_close","reason.needs_delete"]`},
		{ID: 85, Key: "answer.review.reasons", Value: `["reason.looks_ok","reason.needs_edit","reason.needs_delete"]`},
		{ID: 86, Key: "comment.review.reasons", Value: `["reason.looks_ok","reason.needs_edit","reason.needs_delete"]`},
		{ID: 87, Key: "question.asked", Value: `0`},
		{ID: 88, Key: "question.closed", Value: `0`},
		{ID: 89, Key: "question.reopened", Value: `0`},
		{ID: 90, Key: "question.answered", Value: `0`},
		{ID: 91, Key: "question.commented", Value: `0`},
		{ID: 92, Key: "question.accept", Value: `0`},
		{ID: 93, Key: "question.edited", Value: `0`},
		{ID: 94, Key: "question.rollback", Value: `0`},
		{ID: 95, Key: "question.deleted", Value: `0`},
		{ID: 96, Key: "question.undeleted", Value: `0`},
		{ID: 97, Key: "answer.answered", Value: `0`},
		{ID: 98, Key: "answer.commented", Value: `0`},
		{ID: 99, Key: "answer.edited", Value: `0`},
		{ID: 100, Key: "answer.rollback", Value: `0`},
		{ID: 101, Key: "answer.undeleted", Value: `0`},
		{ID: 102, Key: "tag.created", Value: `0`},
		{ID: 103, Key: "tag.edited", Value: `0`},
		{ID: 104, Key: "tag.rollback", Value: `0`},
		{ID: 105, Key: "tag.deleted", Value: `0`},
		{ID: 106, Key: "tag.undeleted", Value: `0`},
		{ID: 107, Key: "rank.comment.vote_up", Value: `1`},
		{ID: 108, Key: "rank.comment.vote_down", Value: `1`},
		{ID: 109, Key: "rank.question.edit_without_review", Value: `2000`},
		{ID: 110, Key: "rank.answer.edit_without_review", Value: `2000`},
		{ID: 111, Key: "rank.tag.edit_without_review", Value: `20000`},
		{ID: 112, Key: "rank.answer.audit", Value: `2000`},
		{ID: 113, Key: "rank.question.audit", Value: `2000`},
		{ID: 114, Key: "rank.tag.audit", Value: `20000`},
		{ID: 115, Key: "rank.question.close", Value: `-1`},
		{ID: 116, Key: "rank.question.reopen", Value: `-1`},
		{ID: 117, Key: "rank.tag.use_reserved_tag", Value: `-1`},
		{ID: 118, Key: "plugin.status", Value: `{}`},
		{ID: 119, Key: "question.pin", Value: `0`},
		{ID: 120, Key: "question.unpin", Value: `0`},
		{ID: 121, Key: "question.show", Value: `0`},
		{ID: 122, Key: "question.hide", Value: `0`},
		{ID: 123, Key: "rank.question.pin", Value: `-1`},
		{ID: 124, Key: "rank.question.unpin", Value: `-1`},
		{ID: 125, Key: "rank.question.show", Value: `-1`},
		{ID: 126, Key: "rank.question.hide", Value: `-1`},
		{ID: 127, Key: "rank.answer.invite_someone_to_answer", Value: `1000`},
		{ID: 128, Key: "rank.answer.undeleted", Value: `-1`},
		{ID: 129, Key: "rank.question.undeleted", Value: `-1`},
		{ID: 130, Key: "rank.tag.undeleted", Value: `-1`},
		{ID: 131, Key: "ai_config.provider", Value: `[{"default_api_host":"https://api.openai.com","display_name":"OpenAI","name":"openai"},{"default_api_host":"https://generativelanguage.googleapis.com","display_name":"Gemini","name":"gemini"},{"default_api_host":"https://api.anthropic.com","display_name":"Anthropic","name":"anthropic"}]`},
	}

	defaultBadgeGroupTable = []*entity.BadgeGroup{
		{ID: "1", Name: "badge.default_badge_groups.getting_started.name"},
		{ID: "2", Name: "badge.default_badge_groups.community.name"},
		{ID: "3", Name: "badge.default_badge_groups.posting.name"},
	}

	defaultBadgeTable = []*entity.Badge{
		{
			Name:         "badge.default_badges.autobiographer.name",
			Icon:         "person-badge-fill",
			Description:  "badge.default_badges.autobiographer.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstUpdateUserProfile",
		},
		{
			Name:         "badge.default_badges.editor.name",
			Icon:         "pencil-fill",
			Description:  "badge.default_badges.editor.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstPostEdit",
		},
		{
			Name:         "badge.default_badges.first_flag.name",
			Icon:         "flag-fill",
			Description:  "badge.default_badges.first_flag.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstFlaggedPost",
		},
		{
			Name:         "badge.default_badges.first_upvote.name",
			Icon:         "hand-thumbs-up-fill",
			Description:  "badge.default_badges.first_upvote.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstVotedPost",
		},
		{
			Name:         "badge.default_badges.first_reaction.name",
			Icon:         "emoji-smile-fill",
			Description:  "badge.default_badges.first_reaction.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstReactedPost",
		},
		{
			Name:         "badge.default_badges.first_share.name",
			Icon:         "share-fill",
			Description:  "badge.default_badges.first_share.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstSharedPost",
		},
		{
			Name:         "badge.default_badges.scholar.name",
			Icon:         "check-circle-fill",
			Description:  "badge.default_badges.scholar.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 1,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "FirstAcceptAnswer",
		},
		{
			Name:         "badge.default_badges.solved.name",
			Icon:         "check-square-fill",
			Description:  "badge.default_badges.solved.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 2,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeSingleAward,
			Handler:      "ReachAnswerAcceptedAmount",
			Param:        `{"amount":"1"}`,
		},
		{
			Name:         "badge.default_badges.nice_answer.name",
			Icon:         "chat-square-text-fill",
			Description:  "badge.default_badges.nice_answer.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 3,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeMultiAward,
			Handler:      "ReachAnswerVote",
			Param:        `{"amount":"10"}`,
		},
		{
			Name:         "badge.default_badges.good_answer.name",
			Icon:         "chat-square-text-fill",
			Description:  "badge.default_badges.good_answer.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 3,
			Level:        entity.BadgeLevelSilver,
			Single:       entity.BadgeMultiAward,
			Handler:      "ReachAnswerVote",
			Param:        `{"amount":"25"}`,
		},
		{
			Name:         "badge.default_badges.great_answer.name",
			Icon:         "chat-square-text-fill",
			Description:  "badge.default_badges.great_answer.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 3,
			Level:        entity.BadgeLevelGold,
			Single:       entity.BadgeMultiAward,
			Handler:      "ReachAnswerVote",
			Param:        `{"amount":"50"}`,
		},
		{
			Name:         "badge.default_badges.nice_question.name",
			Icon:         "question-circle-fill",
			Description:  "badge.default_badges.nice_question.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 3,
			Level:        entity.BadgeLevelBronze,
			Single:       entity.BadgeMultiAward,
			Handler:      "ReachQuestionVote",
			Param:        `{"amount":"10"}`,
		},
		{
			Name:         "badge.default_badges.good_question.name",
			Icon:         "question-circle-fill",
			Description:  "badge.default_badges.good_question.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 3,
			Level:        entity.BadgeLevelSilver,
			Single:       entity.BadgeMultiAward,
			Handler:      "ReachQuestionVote",
			Param:        `{"amount":"25"}`,
		},
		{
			Name:         "badge.default_badges.great_question.name",
			Icon:         "question-circle-fill",
			Description:  "badge.default_badges.great_question.desc",
			Status:       entity.BadgeStatusActive,
			BadgeGroupID: 3,
			Level:        entity.BadgeLevelGold,
			Single:       entity.BadgeMultiAward,
			Handler:      "ReachQuestionVote",
			Param:        `{"amount":"50"}`,
		},
	}
)
